/*

Copyright (c) 2009-2010 Thomas Dennis.

Permission is hereby granted, free of charge, to any person
obtaining a copy of this software and associated documentation
files (the "Software"), to deal in the Software without
restriction, including without limitation the rights to use,
copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following
conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

*/

#ifndef __TGDCSV_H__
#define __TGDCSV_H__

/**
\file  csv.h
\brief Provides utilities for loading and saving CSV spreadsheet files.

Additional formats might be supported in the future.
*/

#ifdef __cplusplus
extern "C" {
#endif

/**
\struct CsvTable
\brief The standard structure used to contain all spreadsheet data.

Consists of a 2D array of NULL-terminated C strings.
*/

typedef struct {

  /** \brief The width of the table in cells. */
  long width;
  /** \brief The height of the table in cells. */
  long height;
  /** \brief Table cell data. */
  char ***cell;

} CsvTable;

/**
\brief Create a CsvTable structure from a CSV-format data buffer.

This function can also be used to import Tab-Seperated or Space-Separated files.

@param len The length of the data buffer.
@param data A raw byte buffer containing file data.
@param cend The col-end character. Usually a comma, but may be a semicolon, etc.
@param qchr The quote character. Usually a doublequote but may be a singlequote.
@return A valid CsvTable on success or NULL on failure.
*/

CsvTable *csvLoadBuffer (long len, const unsigned char *data, char cend,
                                                              char qchr);

/**
\brief Create a CsvTable structure from a file in any supported format.

This function is essentially a special wrapper for csvLoadBuffer(). It creates a
data buffer from a given file, passes it to csvLoadBuffer() and then frees it
before returning the result obtained from the csvLoadBuffer() function.

@param file_name The name and path of the file to be loaded.
@param cend The col-end character. Usually a comma, but may be a semicolon, etc.
@param qchr The quote character. Usually a doublequote but may be a singlequote.
@return A valid CsvTable on success or NULL on failure.
*/

CsvTable *csvLoadFile (const char *file_name, char cend, char qchr);

/**
\brief Create a new CsvTable structure in memory.

Used internally by the various csvLoad* functions.

@param width The width of the table in cells.
@param height The height of the table in cells.
@return A valid CsvTable on success or NULL on failure.
*/

CsvTable *csvCreateTable (long width, long height);

/**
\brief Free a previously allocated CsvTable structure from memory.

After this function has been called, the CsvTable pointer should be set to NULL.

@param csv The table to be freed.
*/

void csvFree (CsvTable *csv);

/**
\brief Export a CsvTable structure as a CSV file.

This function can also be used to export Tab-Seperated or Space-Separated files.

@param csv   The CsvTable to be saved.
@param fname The filename to use for the exported table.
@param cend  The col-end character. Usually a comma but may be a semicolon, etc.
@param qchr  The quote character. Usually a doublequote but may be a singlequote
@return      Zero on success or non-zero on failure.
*/

int csvSaveCSV (const CsvTable *csv, const char *fname, char cend, char qchr);

/**
\brief Export a CsvTable structure as an HTML file.

Exported files are HTML 4.01 strict compliant.

@param csv The CsvTable to be saved.
@param file_name The filename to use for the exported table.
@return Zero on success or non-zero on failure.
*/

int csvSaveHTML (const CsvTable *csv, const char *file_name);

/**
\brief Edit the contents of a CsvTable cell.

Any previous data in the cell will be freed from memory and replaced.

@param csv The CsvTable to be edited.
@param x Cell X position.
@param y Cell Y position.
@param new_text The new value of the cell. This can be set to NULL.
@return Zero on success or non-zero on failure.
*/

int csvEditCell (const CsvTable *csv, long x, long  y, const char *new_text);

/**
\brief Search for a matching string in a particular column of a CsvTable.

Returns the first match found, even when there is more than one possible match.

@param csv The CsvTable to be searched.
@param col The column to search.
@param str The value to look for.
@return a row index on success or a negative value on failure.
*/

long csvSearchCol (const CsvTable *csv, long col, const char *str);

#ifdef __cplusplus
}
#endif

#endif /* __TGDCSV_H__ */
