/*

Copyright (c) 2009-2010 Thomas Dennis.

Permission is hereby granted, free of charge, to any person
obtaining a copy of this software and associated documentation
files (the "Software"), to deal in the Software without
restriction, including without limitation the rights to use,
copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following
conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

*/

#include <stdlib.h>
#include <string.h>
#include "csv.h"

/* Allocate a new table structure in memory. */

CsvTable *csvCreateTable (long width, long height)
{

  CsvTable *csv = NULL;
  long i, j;

  if( width < 1 || height < 1 ) { return NULL; }

  csv = (CsvTable*) malloc( sizeof(CsvTable) );

  if( !csv ) { return NULL; }

  csv->width  = width;
  csv->height = height;

  csv->cell = (char***) malloc( sizeof(char**) * width );

  if( !csv->cell )
  {

    csvFree( csv );

    return NULL;

  }

  for( i = 0; i < width; i++ )
  {

    csv->cell[i] = (char**) malloc( sizeof(char*) * height );

    if( csv->cell[i] )
    {

      for( j = 0; j < height; j++ ) { csv->cell[i][j] = NULL; }

    }

  }

  return csv;

}

/* Free a previously allocated table. */

void csvFree( CsvTable *csv )
{

  long i, j;

  if( !csv ) { return; }

  if( csv->cell )
  {

    for( i = 0; i < csv->width; i++ )
    {

      if( csv->cell[i] )
      {

        for( j = 0; j < csv->height; j++ )
        {

          if( csv->cell[i][j] ) { free( csv->cell[i][j] ); }

        }

        free( csv->cell[i] );

      }

    }

    free( csv->cell );

  }

  free( csv );

}

/* Edit the contents of a CsvTable cell. */

int csvEditCell( const CsvTable *csv, long x, long  y, const char *new_text )
{

  if( !csv ) { return -1; }
  if( x >= csv->width || y >= csv->height ) { return -2; }
  if( !csv->cell      ) { return -3; }
  if( !csv->cell[x] ) { return -4; }

  if( csv->cell[x][y] == new_text ) { return 0; }

  if( csv->cell[x][y] )
  {

    free( csv->cell[x][y] );

    csv->cell[x][y] = NULL;

  }

  if( new_text )
  {

    csv->cell[x][y] = (char*) malloc( strlen( new_text ) + 1 );

    if( csv->cell[x][y] ) { strcpy( csv->cell[x][y], new_text ); }

  }

  return 0;

}

/* Search for a particulare value in a given column. */

long csvSearchCol (const CsvTable *csv, long col, const char *str)
{

  long i = 0;

  if (!csv       || col <  0         ) { return -1; }
  if (!csv->cell || col >= csv->width) { return -1; }

  for (i = 0; i < csv->height; i++)
  {

    if (!str)
    {

      if (!csv->cell[i][col]) { return i; }

    } else if (strcmp (str, csv->cell[col][i]) == 0) { return i; }

  }

  return -1;

}
